/*
 * Copyright (C) 2014 Laboratoire ThéMA - UMR 6049 - CNRS / Université de Franche-Comté
 * http://thema.univ-fcomte.fr
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


package org.thema.graphab.metric.global;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.DoubleAdder;
import org.geotools.graph.structure.Node;
import org.thema.graphab.Project;
import org.thema.graphab.graph.GraphGenerator;
import org.thema.graphab.graph.GraphPathFinder;
import org.thema.graphab.links.Linkset;
import org.thema.graphab.metric.AlphaParamMetric;
import org.thema.graphab.metric.ParamPanel;

/**
 * PC decomposition for delta method.
 * This metric works only in delta mode, for calculating the decomposition of the PC metric:
 * dPCIntra, dPCFlux, dPCConnector
 * @author Gilles Vuidel
 */
public class EChMetric extends AbstractPathMetric {

    private AlphaParamMetric alphaParam = new AlphaParamMetric(false);

    private HashMap<String, DoubleAdder> result;
    
    @Override
    public void startCalc(GraphGenerator g) {
        result = new HashMap<>();
        for(String s : getResultNames(g)) {
            result.put(s, new DoubleAdder());
        }
    }

    @Override
    public Void calcPartMetric(GraphPathFinder finder, GraphGenerator g) {
        double srcCapa = Project.getPatchCapacity(finder.getNodeOrigin());
        int srcCode = Project.getPatchCode(finder.getNodeOrigin());
        for(Node node : finder.getComputedNodes()) { 
            int destCode = Project.getPatchCode(node);
            double f =  srcCapa * Project.getPatchCapacity(node) * Math.exp(-alphaParam.getAlpha()*finder.getCost(node));
            result.get(getName(srcCode, destCode)).add(f);
        }
        
        return null;
    }


    @Override
    public Double[] calcMetric(GraphGenerator g) {
        String[] names = getResultNames(g);
        Double[] res = new Double[names.length];
        int i = 0;
        for(String name : names) {
            res[i++] = Math.sqrt(result.get(name).doubleValue());
        }
        return res;
    }

    @Override
    public String[] getResultNames(GraphGenerator graph) {
        List<Integer> codes = new ArrayList<>(graph.getProject().getPatchCodes());
        List<String> names = new ArrayList<>();
        for(int i = 0; i < codes.size(); i++) {
            for(int j = i; j < codes.size(); j++) {
                names.add(getName(codes.get(i), codes.get(j)));
            }
        }
        return names.toArray(new String[names.size()]);
    }

    private String getName(int code1, int code2) {
        return Math.min(code1, code2) + "/" + Math.max(code1, code2);
    }
    
    @Override
    public String getShortName() {
        return "ECh";
    }
    
    
    @Override
    public void setParams(Map<String, Object> params) {
        alphaParam.setParams(params);
    }

    @Override
    public LinkedHashMap<String, Object> getParams() {
        return alphaParam.getParams();
    }

    @Override
    public ParamPanel getParamPanel(Linkset linkset) {
        return alphaParam.getParamPanel(linkset);
    }
    
    @Override
    public Type getType() {
        return Type.WEIGHT;
    }

    @Override
    public boolean isAcceptGraph(GraphGenerator graph) {
        return super.isAcceptGraph(graph) && !(graph.getProject().isMerge() && graph.getProject().getPatchCodes().size() > 1);
    }

    @Override
    public void mergePart(Object part) {
        // does nothing
    }
}
